#include <ESP8266WiFi.h>
#include <ESP8266WebServer.h>
#include <EEPROM.h>
#include <WiFiUdp.h>

// ---------------- PIN DEFINITIONS ----------------
#define TRIG_PIN       D6
#define ECHO_PIN       D7
#define WIFI_LED_PIN   D2
#define AUTO_LED_PIN   D3
#define MANUAL_LED_PIN D4
#define RELAY_PIN      D5

// ---------------- GLOBAL OBJECTS ----------------
ESP8266WebServer server(80);
WiFiUDP udp;
unsigned int port = 4210;

#define EEPROM_SIZE 192

// ---------------- TANK VARIABLES ----------------
float tankHeight = 40;
float highLevelPercent = 90;
float lowLevelPercent  = 30;
bool motorState = false;
bool autoMode = true;

float filteredLevel = 0;
int lastLevel = -1;
unsigned long lastLevelChange = 0;
unsigned long lastCheck = 0;

const unsigned long DRY_RUN_TIMEOUT = 60UL * 1000UL;  // 1 minute
bool dryRunTriggered = false;

// ---------------- ROUTER CREDENTIALS ----------------
char routerSSID[32] = "";
char routerPASS[64] = "";
bool staConfigured = false;

// ------------------------------------------------------
//                       ULTRASONIC
// ------------------------------------------------------
float getDistance() {
  float sum = 0;
  for (int i = 0; i < 5; i++) {
    digitalWrite(TRIG_PIN, LOW);
    delayMicroseconds(2);
    digitalWrite(TRIG_PIN, HIGH);
    delayMicroseconds(10);
    digitalWrite(TRIG_PIN, LOW);

    float duration = pulseIn(ECHO_PIN, HIGH, 25000);
    float d = duration * 0.0343 / 2.0;
    if (d < 2 || d > 500) d = tankHeight;

    sum += d;
    delay(10);
    yield();
  }
  return sum / 5;
}

float getLevelPercent() {
  float distance = getDistance();
  float level = 100 - ((distance / tankHeight) * 100);
  level = constrain(level, 0, 100);

  if (level > 95) {
    filteredLevel = 100;
    return 100;
  }

  filteredLevel = (0.8 * filteredLevel) + (0.2 * level);
  return filteredLevel;
}

// ------------------------------------------------------
//                       EEPROM
// ------------------------------------------------------
void saveSettings() {
  EEPROM.put(0, tankHeight);
  EEPROM.put(10, highLevelPercent);
  EEPROM.put(20, lowLevelPercent);
  EEPROM.put(30, autoMode);
  EEPROM.put(40, motorState);
  EEPROM.put(50, routerSSID);
  EEPROM.put(90, routerPASS);
  EEPROM.commit();
}

void loadSettings() {
  EEPROM.get(0, tankHeight);
  EEPROM.get(10, highLevelPercent);
  EEPROM.get(20, lowLevelPercent);
  EEPROM.get(30, autoMode);
  EEPROM.get(40, motorState);
  EEPROM.get(50, routerSSID);
  EEPROM.get(90, routerPASS);

  if (tankHeight < 5 || tankHeight > 500) tankHeight = 40;
  if (highLevelPercent > 100) highLevelPercent = 90;
  if (lowLevelPercent < 0) lowLevelPercent = 30;

  staConfigured = routerSSID[0] != '\0';
}

// ------------------------------------------------------
//                     MOTOR CONTROL
// ------------------------------------------------------
void motorOn() {
  digitalWrite(RELAY_PIN, HIGH);
  motorState = true;
  dryRunTriggered = false;

  lastLevel = getLevelPercent();
  lastLevelChange = millis();

  saveSettings();
}

void motorOff() {
  digitalWrite(RELAY_PIN, LOW);
  motorState = false;
  saveSettings();
}

void updateMotorState(float level) {
  if (autoMode && !dryRunTriggered) {
    if (level <= lowLevelPercent && !motorState) motorOn();
    if (level >= highLevelPercent && motorState) motorOff();
  }
}

void updateModeLEDs() {
  digitalWrite(AUTO_LED_PIN, autoMode);
  digitalWrite(MANUAL_LED_PIN, !autoMode);
}

// ------------------------------------------------------
//                SUPER STABLE WiFi SYSTEM
// ------------------------------------------------------
extern "C" {
  #include "user_interface.h"
}

void setupWiFiStable()
{
  Serial.println("\n🚀 Setting up SUPER-STABLE WiFi...");

  WiFi.persistent(false);
  WiFi.setAutoReconnect(true);

  wifi_station_set_auto_connect(true);
  wifi_station_set_reconnect_policy(true);

  // ⭐ Modern sleep disable function
  WiFi.setSleepMode(WIFI_NONE_SLEEP);

  WiFi.mode(WIFI_AP_STA);

  // 1) Start AP on fixed channel 6
  WiFi.softAP("SmartWaterLevelController", "12345678", 6, false);

  Serial.print("AP Started → IP: ");
  Serial.println(WiFi.softAPIP());

  // 2) Now connect STA
  WiFi.begin(routerSSID, routerPASS);

  unsigned long start = millis();
  while (WiFi.status() != WL_CONNECTED && millis() - start < 20000) {
    Serial.print(".");
    delay(300);
  }
  Serial.println("");

  if (WiFi.status() == WL_CONNECTED) {

    // 3) Match AP channel with router
    int ch = WiFi.channel();
    wifi_set_channel(ch);

    WiFi.softAP("SmartWaterLevelController", "12345678", ch, false);

    Serial.println("🌐 STA Connected!");
    Serial.print("STA IP: ");
    Serial.println(WiFi.localIP());

    Serial.print("Router Channel Detected = ");
    Serial.println(ch);

  } else {
    Serial.println("❌ STA Failed → Staying AP only.");
  }
}


// ------------------------------------------------------
//                 /setWiFi (Update Router)
// ------------------------------------------------------
void handleSetWiFi() {
  if (!server.hasArg("ssid") || !server.hasArg("pass")) {
    server.send(400, "text/plain", "Missing SSID or PASSWORD");
    return;
  }

  String ssid = server.arg("ssid");
  String pass = server.arg("pass");

  memset(routerSSID, 0, sizeof(routerSSID));
  memset(routerPASS, 0, sizeof(routerPASS));

  ssid.toCharArray(routerSSID, sizeof(routerSSID));
  pass.toCharArray(routerPASS, sizeof(routerPASS));

  staConfigured = true;
  saveSettings();

  server.send(200, "text/plain", "OK");

  ESP.restart();
}

// ------------------------------------------------------
//                     HTTP ROUTES
// ------------------------------------------------------
bool isAPModeOnly() {
  IPAddress clientIP = server.client().remoteIP();

  // If app is connected to NodeMCU hotspot (192.168.4.x)
  if (clientIP[0] == 192 && clientIP[1] == 168 && clientIP[2] == 4) {
    return true;   // AP mode
  }

  return false;    // Router mode
}


void handleMotorState() {
  if (dryRunTriggered)
    server.send(200, "text/plain", "DryRun");
  else
    server.send(200, "text/plain", motorState ? "1" : "0");
}

void handleLevel() {
  // AP mode में level भी नहीं देंगे (sirf STA में full control)
  if (isAPModeOnly()) {
    server.send(403, "text/plain", "AP_MODE_LIMITED");
    return;
  }

  float level = getLevelPercent();

  // Yaha auto mode ka logic chalta rahega
  updateMotorState(level);

  server.send(200, "text/plain", String(level, 1));
}


void handleMotorOn() {
  // 1) AP mode में block
  if (isAPModeOnly()) {
    server.send(403, "text/plain", "AP_MODE_LIMITED");
    return;
  }

  // 2) Auto mode में manual ON allowed नहीं
  if (autoMode == true) {
    server.send(403, "text/plain", "MANUAL_REQUIRED");
    return;
  }

  motorOn();
  server.send(200, "text/plain", "1");
}


void handleMotorOff() {
  // 1) AP mode में block
  if (isAPModeOnly()) {
    server.send(403, "text/plain", "AP_MODE_LIMITED");
    return;
  }

  // 2) Auto mode में manual OFF allowed नहीं
  if (autoMode == true) {
    server.send(403, "text/plain", "MANUAL_REQUIRED");
    return;
  }

  motorOff();
  server.send(200, "text/plain", "0");
}


void handleUpdate() {
  // AP mode में koi bhi setting change नहीं होगी
  if (isAPModeOnly()) {
    server.send(403, "text/plain", "AP_MODE_LIMITED");
    return;
  }

  if (server.hasArg("height")) tankHeight = server.arg("height").toFloat();
  if (server.hasArg("low"))    lowLevelPercent = server.arg("low").toFloat();
  if (server.hasArg("high"))   highLevelPercent = server.arg("high").toFloat();
  if (server.hasArg("mode"))   autoMode = (server.arg("mode") == "1");

  updateModeLEDs();
  saveSettings();

  server.send(200, "text/plain", "OK");
}



void handleGetSettings() {
  // AP mode में full settings देना allowed नहीं
  if (isAPModeOnly()) {
    server.send(403, "text/plain", "AP_MODE_LIMITED");
    return;
  }

  String data =
      String(tankHeight) + "," +
      String(lowLevelPercent) + "," +
      String(highLevelPercent) + "," +
      (autoMode ? "1" : "0") + "," +
      (motorState ? "1" : "0");

  server.send(200, "text/plain", data);
}


void handleClearDryRun() {
  dryRunTriggered = false;
  motorOff();
  server.send(200, "text/plain", "OK");
}

// ------------------------------------------------------
//                        SETUP
// ------------------------------------------------------
void setup() {
  Serial.begin(115200);
  EEPROM.begin(EEPROM_SIZE);

  pinMode(TRIG_PIN, OUTPUT);
  pinMode(ECHO_PIN, INPUT);
  pinMode(RELAY_PIN, OUTPUT);
  pinMode(WIFI_LED_PIN, OUTPUT);
  pinMode(AUTO_LED_PIN, OUTPUT);
  pinMode(MANUAL_LED_PIN, OUTPUT);

  loadSettings();
  digitalWrite(RELAY_PIN, motorState);

  updateModeLEDs();

  setupWiFiStable();

  udp.begin(port);

  server.on("/level", handleLevel);
  server.on("/motor/state", handleMotorState);
  server.on("/motor/on", handleMotorOn);
  server.on("/motor/off", handleMotorOff);
  server.on("/update", handleUpdate);
  server.on("/getSettings", handleGetSettings);
  server.on("/setWiFi", handleSetWiFi);
  server.on("/clearDryRun", handleClearDryRun);
  server.on("/getRouterSSID", []() {
  server.send(200, "text/plain", WiFi.SSID());
});


  server.begin();
  Serial.println("HTTP Server started");
}

// ------------------------------------------------------
//                        LOOP
// ------------------------------------------------------
void loop() {
  server.handleClient();

  // -------- UDP Discovery --------
  int size = udp.parsePacket();
  if (size) {
    char msg[50];
    int len = udp.read(msg, 49);
    msg[len] = 0;

    if (String(msg).startsWith("DISCOVER_NODEMCU")) {
      IPAddress ip = (WiFi.status() == WL_CONNECTED) ? WiFi.localIP() : WiFi.softAPIP();
      String r = "NODEMCU_OK,IP=" + ip.toString() + ",ID=Tank1";

      udp.beginPacket(udp.remoteIP(), udp.remotePort());
      udp.print(r);
      udp.endPacket();
    }
  }

  // -------- DRY RUN DETECTION --------
  unsigned long now = millis();

  if (motorState && !dryRunTriggered) {
    float level = getLevelPercent();

    if (abs(level - lastLevel) > 1) {
      lastLevel = level;
      lastLevelChange = now;
    }
    else if (now - lastLevelChange > DRY_RUN_TIMEOUT) {
      dryRunTriggered = true;
      motorOff();
    }
  }

  // -------- WiFi LED --------
  if (WiFi.status() == WL_CONNECTED || WiFi.softAPgetStationNum() > 0)
    digitalWrite(WIFI_LED_PIN, HIGH);
  else
    digitalWrite(WIFI_LED_PIN, (now / 500) % 2);
}
